import express from 'express';
import { readFileSync, writeFileSync, existsSync, watch } from 'fs';
import { fileURLToPath } from 'url';
import { dirname, join } from 'path';
import cookieParser from 'cookie-parser';

// ===== CONFIGURATION =====
const AUTH_PASSWORD = 'changeme';       // Change this to your password
const PORT = 3000;                       // Change for multiple apps: 3000, 3001, 3002...
const DB_FILE = 'db.json';              // Database file name
// =========================

const __dirname = dirname(fileURLToPath(import.meta.url));
const app = express();
const DB_PATH = join(__dirname, DB_FILE);

// Initialize db.json if it doesn't exist
if (!existsSync(DB_PATH)) {
  writeFileSync(DB_PATH, JSON.stringify({ notes: [] }, null, 2));
  console.log('Created db.json');
}

// Middleware
app.use(express.json({ limit: '1mb' }));
app.use(express.urlencoded({ extended: true }));
app.use(cookieParser());

// Login page
app.get('/login', (req, res) => {
  res.sendFile(join(__dirname, 'login.html'));
});

// Login handler
app.post('/login', (req, res) => {
  if (req.body.password === AUTH_PASSWORD) {
    res.cookie('auth', 'valid', {
      httpOnly: true,
      maxAge: 365 * 24 * 60 * 60 * 1000 // 1 year
    });
    res.redirect('/');
  } else {
    res.redirect('/login');
  }
});

// Auth check middleware
app.use((req, res, next) => {
  if (req.path === '/login' || req.cookies.auth === 'valid') {
    next();
  } else {
    res.redirect('/login');
  }
});

// Serve index.html for root
app.get('/', (req, res) => {
  res.sendFile(join(__dirname, 'index.html'));
});

// Read notes
app.get('/api/notes', (req, res) => {
  try {
    const data = JSON.parse(readFileSync(DB_PATH, 'utf8'));
    res.json(data);
  } catch (err) {
    res.json({ notes: [] });
  }
});

// Write notes
app.post('/api/notes', (req, res) => {
  try {
    writeFileSync(DB_PATH, JSON.stringify(req.body, null, 2));
    notifyClients(); // Notify all connected clients
    res.json({ success: true });
  } catch (err) {
    res.status(500).json({ error: 'Failed to save' });
  }
});

// SSE endpoint for real-time updates
let clients = [];

app.get('/api/updates', (req, res) => {
  res.setHeader('Content-Type', 'text/event-stream');
  res.setHeader('Cache-Control', 'no-cache');
  res.setHeader('Connection', 'keep-alive');

  // Send initial heartbeat
  res.write(':heartbeat\n\n');

  // Add this client to the list
  clients.push(res);

  // Remove client when connection closes
  req.on('close', () => {
    clients = clients.filter(client => client !== res);
  });
});

// Send heartbeat every 30 seconds to keep connections alive
setInterval(() => {
  clients.forEach(client => {
    try {
      client.write(':heartbeat\n\n');
    } catch (err) {
      // Client disconnected, will be cleaned up
    }
  });
  // Clean up dead clients
  clients = clients.filter(c => !c.writableEnded);
}, 30000);

function notifyClients() {
  clients.forEach(client => {
    try {
      client.write('data: update\n\n');
    } catch (err) {
      // Client disconnected, ignore
    }
  });
  // Clean up dead clients
  clients = clients.filter(c => !c.writableEnded);
}

// Start server
app.listen(PORT, () => {
  console.log(`Server running on http://localhost:${PORT}`);
});